#!/usr/bin/env bash
set -euo pipefail

REPO_URL="https://github.com/guardrail-labs/llm-guardrail-api-next.git"
TARGET_DIR_NAME="llm-guardrail-api-next"

# On Windows/Git Bash, long paths can break venv creation (WinError 206).
# Use a short workspace root to avoid MAX_PATH issues.
SHORT_ROOT="/c/guardrail"
TARGET_DIR_DEFAULT="./${TARGET_DIR_NAME}"

uname_out="$(uname -s || true)"
case "${uname_out}" in
  MINGW*|MSYS*|CYGWIN*)
    TARGET_DIR="${SHORT_ROOT}/${TARGET_DIR_NAME}"
    ;;
  *)
    TARGET_DIR="${TARGET_DIR_DEFAULT}"
    ;;
esac

echo ""
echo "Guardrail API — Student / Free Bootstrap"
echo "----------------------------------------"
echo ""
echo "This script will:"
echo "  1) Clone the Guardrail Core Runtime repo"
echo "  2) Create a Python virtual environment (.venv)"
echo "  3) Install Guardrail Core for local use"
echo ""

# Detect available Python command
# Detect available Python command (Windows Git Bash often has a broken python3 alias)
PYTHON_CMD=""

uname_out="$(uname -s || true)"
case "${uname_out}" in
  MINGW*|MSYS*|CYGWIN*)
    # On Windows, prefer "python" (python3 may be a Microsoft Store stub)
    if command -v python >/dev/null 2>&1; then
      PYTHON_CMD="python"
    elif command -v python3 >/dev/null 2>&1; then
      PYTHON_CMD="python3"
    fi
    ;;
  *)
    # On macOS/Linux, prefer python3
    if command -v python3 >/dev/null 2>&1; then
      PYTHON_CMD="python3"
    elif command -v python >/dev/null 2>&1; then
      PYTHON_CMD="python"
    fi
    ;;
esac

if [ -z "${PYTHON_CMD}" ]; then
  echo "ERROR: Python not found."
  echo "Install Python 3.11+ from https://www.python.org/downloads/"
  exit 1
fi


PYTHON_VERSION="$(${PYTHON_CMD} -c 'import sys; print(f"{sys.version_info.major}.{sys.version_info.minor}")')"

if ! ${PYTHON_CMD} -c "import sys; raise SystemExit(0 if sys.version_info >= (3,11) else 1)" >/dev/null 2>&1; then
  echo "ERROR: Python 3.11 or higher is required."
  echo "Current Python: ${PYTHON_CMD} (${PYTHON_VERSION})"
  exit 1
fi

echo "Using Python: ${PYTHON_CMD} ($(${PYTHON_CMD} --version 2>&1))"
echo ""

# Ensure short root exists on Windows
if [[ "${TARGET_DIR}" == "${SHORT_ROOT}/"* ]]; then
  mkdir -p "${SHORT_ROOT}"
  echo "Using short workspace path (Windows compatibility):"
  echo "  ${TARGET_DIR}"
  echo ""
fi

# Clone repo if needed
if [ -d "${TARGET_DIR}" ]; then
  echo "Repository directory already exists. Skipping clone:"
  echo "  ${TARGET_DIR}"
else
  echo "Cloning Guardrail Core Runtime into:"
  echo "  ${TARGET_DIR}"
  git clone "${REPO_URL}" "${TARGET_DIR}"
fi

cd "${TARGET_DIR}"

# Create venv if needed (and recreate if mismatched Python version)
if [ -d ".venv" ]; then
  VENV_PYTHON_VERSION="unknown"
  if [ -f ".venv/bin/python" ]; then
    VENV_PYTHON_VERSION=$(.venv/bin/python -c 'import sys; print(f"{sys.version_info.major}.{sys.version_info.minor}")' 2>/dev/null || echo "unknown")
  elif [ -f ".venv/Scripts/python.exe" ]; then
    VENV_PYTHON_VERSION=$(.venv/Scripts/python.exe -c 'import sys; print(f"{sys.version_info.major}.{sys.version_info.minor}")' 2>/dev/null || echo "unknown")
  fi

  if [ "${VENV_PYTHON_VERSION}" != "unknown" ] && [ "${VENV_PYTHON_VERSION}" != "${PYTHON_VERSION}" ]; then
    echo "Existing virtual environment uses Python ${VENV_PYTHON_VERSION}, system has ${PYTHON_VERSION}."
    echo "Recreating .venv with Python ${PYTHON_VERSION}..."
    rm -rf .venv
    ${PYTHON_CMD} -m venv .venv
  else
    echo "Virtual environment '.venv' already exists. Skipping creation."
  fi
else
  echo "Creating Python virtual environment in .venv ..."
  ${PYTHON_CMD} -m venv .venv
fi

# Activate venv (Windows vs macOS/Linux)
echo "Activating virtual environment ..."

if [ -f ".venv/Scripts/activate" ]; then
  # Windows (Git Bash)
  # shellcheck disable=SC1091
  source .venv/Scripts/activate
elif [ -f ".venv/bin/activate" ]; then
  # macOS / Linux
  # shellcheck disable=SC1091
  source .venv/bin/activate
else
  echo "ERROR: Could not find a virtualenv activate script under:"
  echo "  $(pwd)/.venv"
  exit 1
fi

python -m pip install -U pip

echo "Installing Guardrail Core (editable) ..."
pip install -e .

echo ""
echo "Install complete."
echo ""
echo "Next steps (copy/paste):"
echo ""
echo "  cd \"$(pwd)\""
if [ -f ".venv/Scripts/activate" ]; then
  echo "  source .venv/Scripts/activate"
else
  echo "  source .venv/bin/activate"
fi
echo "  uvicorn app.main:app --host 127.0.0.1 --port 8000"
echo ""
echo "Then open:"
echo "  http://127.0.0.1:8000/student"
echo ""
echo "Optional API docs (advanced):"
echo "  http://127.0.0.1:8000/docs"

